// entities.c

#include "light.h"

entity_t	entities[MAX_MAP_ENTITIES];
int			num_entities;

/*
==============================================================================

ENTITY FILE PARSING

If a light has a targetname, generate a unique style in the 32-63 range
==============================================================================
*/

#define MAX_LIGHT_TARGETS 32

int	numlighttargets;
char	lighttargets[MAX_LIGHT_TARGETS][MAX_VALUE];

int LightStyleForTargetname (char *targetname, qboolean alloc)
{
	int		i;
	
	for (i=0 ; i<numlighttargets ; i++)
		if (!strcmp (lighttargets[i], targetname))
			return 32 + i;
	if (!alloc)
		return -1;

	if (i == MAX_LIGHT_TARGETS)
		Error("LightStyleForTargetname: Too many unique light targetnames, max = %i", MAX_LIGHT_TARGETS);

	strcpy (lighttargets[i], targetname);
	numlighttargets++;
	return numlighttargets-1 + 32;
}


/*
==================
MatchTargets
==================
*/
void MatchTargets (void)
{
	entity_t *e;
	int	 i,j;
	char	 TargetName[2 * MAX_VALUE], Entity[2 * MAX_VALUE];
	
	for (i=0 ; i<num_entities ; i++)
	{
		e = &entities[i];

		if (e->target == NULL)
			continue;
			
		for (j=0 ; j<num_entities ; j++)
		{
			if (entities[j].targetname == NULL)
				continue;

			if (!strcmp(entities[j].targetname, e->target))
			{
				e->targetent = &entities[j];
				break;
			}
		}

		if (j==num_entities)
		{
			TargetName[0] = '\0';
			
			if (e->targetname != NULL)
				sprintf(TargetName, ", targetname '%s'", e->targetname);

			if (e->origin_set)
				sprintf(Entity, "at (%.0f %.0f %.0f) (%s%s)", e->origin[0], e->origin[1], e->origin[2], e->classname, TargetName);
			else
				sprintf(Entity, "%s%s", e->classname, TargetName);

			logprintf ("WARNING: Entity %s has unmatched target '%s'\n", Entity, e->target);
			continue;
		}
		
// set the style on the source ent for switchable lights
		if (entities[j].style)
		{
			char	s[16];
			
			e->style = entities[j].style;
			sprintf (s,"%i", e->style);
			SetKeyValue (e, "style", s);
		}
	}	
}

/*
==================
FixMinMax
==================
*/
void FixMinMax(int EntLight, qboolean Min)
{
	int	 *PWLight, MinValue;
	qboolean CmdLine;

	PWLight = Min ? &worldminlight : &worldmaxlight;
	MinValue = Min ? 1 : 0;

	CmdLine = *PWLight >= 0;

	if (!CmdLine && EntLight >= MinValue)
		*PWLight = EntLight;
	
	if (*PWLight >= 0)
		logprintf("Using %slight value %i from %s\n", Min ? "min" : "max", *PWLight, CmdLine ? "command line" : "worldspawn");
}

/*
==================
LoadEntities
==================
*/
void LoadEntities (void)
{
	char 		*data;
	entity_t	*entity;
        char		key[MAX_KEY];
	epair_t		*epair;
	double		vec[3];
	int		i, StrangeKeys, UnsupportedKeys, UnsupportedEnts; 
	int		num_lights, TyrLiteKeys, TyrLiteEnts, IKLiteEnts;
	qboolean	ForceMin;
 
	data = dentdata;
//
// start parsing
//
	num_entities = 0;
	num_lights = 0;

	TyrLiteKeys = TyrLiteEnts = IKLiteEnts = 0;

// go through all the entities
	while (1)
	{
	// parse the opening brace	
		data = COM_Parse (data);
		if (!data)
			break;
		if (com_token[0] != '{')
			Error ("LoadEntities: found %s when expecting {",com_token);

		if (num_entities == MAX_MAP_ENTITIES)
			Error ("LoadEntities: MAX_MAP_ENTITIES");
		entity = &entities[num_entities];
		entity->maxlight = -1;
		num_entities++;
		
		StrangeKeys = 0;

	// go through all the keys in this entity
		while (1)
		{
			int		c;

		// parse key
			data = COM_Parse (data);
			if (!data)
				Error ("LoadEntities: EOF without closing brace");
			if (!strcmp(com_token,"}"))
				break;
			if (strlen(com_token) >= MAX_KEY)
				Error ("LoadEntities: Key length > %i", MAX_KEY - 1);
			strcpy (key, com_token);

		// parse value
			data = COM_Parse (data);
			if (!data)
				Error ("LoadEntities: EOF without closing brace");
			c = com_token[0];
			if (c == '}')
				Error ("LoadEntities: closing brace without data");
			if (strlen(com_token) >= MAX_VALUE)
				Error("LoadEntities: Value length > %i", MAX_VALUE - 1);
			
			epair = malloc (sizeof(epair_t));
			memset (epair, 0, sizeof(epair));
			epair->key = copystring(key);
			epair->value = copystring(com_token);
			epair->next = entity->epairs;
			entity->epairs = epair;
			
			if (!strcmp(key, "classname"))
				entity->classname = copystring(com_token);
			else if (!strcmp(key, "target"))
				entity->target = copystring(com_token);
			else if (!strcmp(key, "targetname"))
				entity->targetname = copystring(com_token);
			else if (!strcmp(key, "origin"))
			{
				// scan into doubles, then assign
				// which makes it vec_t size independent
				if (sscanf(com_token, "%lf %lf %lf",
						&vec[0], &vec[1], &vec[2]) != 3)
					Error ("LoadEntities: not 3 values for origin");
				for (i=0 ; i<3 ; i++)
					entity->origin[i] = vec[i];
				
				entity->origin_set = true;
			}
			else if (!strncmp(key, "light", 5) || !strcmp (key, "_light") )
			{
				entity->light = atoi(com_token);
				
				if (entity->light < 0)
					++StrangeKeys; // Negative light
			}
			else if (!strcmp(key, "maxlight"))
				entity->maxlight = atoi(com_token);
			else if (!strcmp(key, "style"))
			{
				entity->style = atoi(com_token);
				if ((unsigned)entity->style > 254)
					Error ("Bad light style %i (must be 0-254)", entity->style);
			}
			else if (!strcmp(key, "angle"))
			{
				entity->angle = atof(com_token);
			}
			else if (!strcmp(key, "wait") ||
				 !strcmp(key, "delay") ||
				 !strcmp(key, "mangle") ||
				 !strcmp(key, "_sunlight") ||
				 !strcmp(key, "_sun_mangle"))
				++StrangeKeys;
		
		}

	// all fields have been parsed
		if (!strncmp (entity->classname, "light", 5))
		{
			if (!entity->light)
				entity->light = DEFAULTLIGHTLEVEL;
	
			if (StrangeKeys != 0)
			{
				TyrLiteKeys += StrangeKeys;
				++TyrLiteEnts;
			}

			if (IKLiteEnts != -1)
			{
				if (entity->angle > 3)
					IKLiteEnts = -1; // Probably not IKLite
				else if (entity->angle >= 1 && entity->angle <= 3)
					++IKLiteEnts;
			}

			num_lights++; 
		}

		if (!strcmp (entity->classname, "light"))
		{
			if (entity->targetname != NULL && !entity->style)
			{
				char	s[16];
				
				entity->style = LightStyleForTargetname (entity->targetname, true);
				sprintf (s,"%i", entity->style);
				SetKeyValue (entity, "style", s);
			}
		}

		if (!strcmp (entity->classname, "worldspawn"))
		{
			FixMinMax(entity->light, true);
			FixMinMax(entity->maxlight, false);

			ForceMin = false;

			if (worldminlight > worldmaxlight && worldmaxlight >= 0)
			{
				worldminlight = worldmaxlight; // Maxlight overrides minlight
				ForceMin = true;
			}

			if (nolight && worldminlight <= 0)
			{
				worldminlight = 1; // Otherwise map will be fullbright
				ForceMin = true;
			}

			if (ForceMin)
				logprintf ("Forcing minlight value %d\n", worldminlight); 

			if (StrangeKeys != 0)
			{
				TyrLiteKeys += StrangeKeys;
				++TyrLiteEnts;
			}
		}
		else
		{
			if (nolight)
				entity->light = 0; // Disable all other light entities
		}
	}

	if (TyrLiteEnts > 0 || IKLiteEnts > 0)
	{
		if (IKLiteEnts > 2 * TyrLiteEnts)
			UnsupportedKeys = UnsupportedEnts = IKLiteEnts;
		else
		{
			UnsupportedKeys = TyrLiteKeys;
			UnsupportedEnts = TyrLiteEnts;
		}
	
		logprintf ("WARNING: %d unsupported light keys found in %d entities, %s recommended\n",
		           UnsupportedKeys, UnsupportedEnts, IKLiteEnts > 2 * TyrLiteEnts ? "IKLite" : "TyrLite");
	}

	logprintf ("%d entities read, %d are lights\n", num_entities, num_lights); 
	MatchTargets ();
}

char 	*ValueForKey (entity_t *ent, char *key)
{
	epair_t	*ep;
	
	for (ep=ent->epairs ; ep ; ep=ep->next)
		if (!strcmp (ep->key, key) )
			return ep->value;
	return "";
}

void 	SetKeyValue (entity_t *ent, char *key, char *value)
{
	epair_t	*ep;
	
	for (ep=ent->epairs ; ep ; ep=ep->next)
		if (!strcmp (ep->key, key) )
		{
			free (ep->value);
			ep->value = copystring(value);
			return;
		}
	ep = malloc (sizeof(*ep));
	ep->next = ent->epairs;
	ent->epairs = ep;
	ep->key = copystring(key);
	ep->value = copystring(value);
}

float	FloatForKey (entity_t *ent, char *key)
{
	char	*k;
	
	k = ValueForKey (ent, key);
	return atof(k);
}

void 	GetVectorForKey (entity_t *ent, char *key, vec3_t vec)
{
	char	*k;
	double	v1, v2, v3;

	k = ValueForKey (ent, key);
	v1 = v2 = v3 = 0;
// scanf into doubles, then assign, so it is vec_t size independent
	sscanf (k, "%lf %lf %lf", &v1, &v2, &v3);
	vec[0] = v1;
	vec[1] = v2;
	vec[2] = v3;
}



/*
================
WriteEntitiesToString
================
*/
void WriteEntitiesToString (void)
{
	char	*buf, *end;
	epair_t	*ep;
	char	line[MAX_KEY + MAX_VALUE + 10];
	int		i, len;
	
	buf = dentdata;
	end = buf;
	*end = 0;
	
	logprintf ("%i switchable light styles\n", numlighttargets);
	
	for (i=0 ; i<num_entities ; i++)
	{
		ep = entities[i].epairs;
		if (!ep)
			continue;	// ent got removed
		
		strcat (end,"{\n");
		end += 2;
				
		for (ep = entities[i].epairs ; ep ; ep=ep->next)
		{
			len = sprintf (line, "\"%s\" \"%s\"\n", ep->key, ep->value);

			if (len > 128)
			{
				strcpy(&line[128 - 2], "\"\n"); // Cut off string at 128
				len = 128;
			}

			strcat (end, line);
			end += len;
		}
		strcat (end,"}\n");
		end += 2;

		if (end > buf + MAX_MAP_ENTSTRING)
			Error ("Entity text too long");
	}
	entdatasize = end - buf + 1;
}

