// util.c

#include <conio.h>
#include <stdarg.h>
#include <malloc.h>
#include "qbsp.h"

extern char *rgszWarnings[cWarnings];
extern char *rgszErrors[cErrors];

int rgMemActive[GLOBAL+1];
int rgMemPeak[GLOBAL+1];

/*
==========
AllocMem
==========
*/
void *AllocMem (int Type, int cElements, bool fZero)
{
	void *pTemp;
	int cSize;

	if (Type < 0 || Type > OTHER)
		Message(msgError, errInvalidMemType, Type);

	// For windings, cElements == number of points on winding
	if (Type == WINDING)
	{
		if (cElements > MAX_POINTS_ON_WINDING)
			Message(msgError, errTooManyPoints, cElements);

		cSize = (int)((winding_t *)0)->points[cElements];

		// Set cElements to 1 so bookkeeping works OK
		cElements = 1;

		// Allocate extra storage for size
		cSize += sizeof(int);
	}
	else
		cSize = cElements * rgcMemSize[Type];

Retry:
	pTemp = (void *)malloc(cSize);

	if (pTemp == NULL)
	{
		char ch;

		printf("\bOut of memory.  Please close some apps and hit 'y' to try again,\n");
		printf("or any other key to exit -> ");
		ch = toupper(_getche());
		printf("\n");
		fflush(stdin);
		if (ch == 'Y')
			goto Retry;
		Message(msgError, errOutOfMemory);
	}

	if (fZero)
		memset(pTemp, 0, cSize);

	// Special stuff for face_t
	if (Type == FACE)
		((face_t *)pTemp)->planenum = -1;
	else if (Type == WINDING)
	{
		// Save size just before the user area
		*(int *)pTemp = cSize;
		pTemp = (char *)pTemp + sizeof(int);
	}

	rgMemActive[Type] += cElements;
	if (rgMemActive[Type] > rgMemPeak[Type])
		rgMemPeak[Type] = rgMemActive[Type];

	// Also keep global statistics
	rgMemActive[GLOBAL] += cSize;
	if (rgMemActive[GLOBAL] > rgMemPeak[GLOBAL])
		rgMemPeak[GLOBAL] = rgMemActive[GLOBAL];

	return pTemp;
}


/*
==========
FreeMem
==========
*/
void FreeMem (void *pMem, int Type, int cElements)
{
	rgMemActive[Type] -= cElements;

	if (Type == FACE)
		ResizeFace ((face_t *)pMem, 0);

	if (Type == WINDING)
	{
		// Size is saved just before the user area
		pMem = (char *)pMem - sizeof(int);
		rgMemActive[GLOBAL] -= *(int *)pMem;
	}
	else
	        rgMemActive[GLOBAL] -= cElements * rgcMemSize[Type];

	free(pMem);
}

void ResizeFace (face_t *f, int numpoints)
{
	// Note: f->numpoints may not be correct here
	f->pts = (vec3_t *)ReAllocOther(f->pts, numpoints * sizeof(vec3_t));
	f->numpoints = numpoints;
}

void CopyFace (face_t *dst, face_t *src)
{
	vec3_t *Save;

	ResizeFace (dst, src->numpoints);

	Save = dst->pts;
	*dst = *src;
	dst->pts = Save;

	memcpy(dst->pts, src->pts, dst->numpoints * sizeof(vec3_t));
}

void *ReAllocOther (void *o, size_t Size)
{
	void *o2;
	int  OldSize;

	o2 = Size == 0 ? NULL : AllocMem (OTHER, Size + sizeof(int));

	if (o2 != NULL)
	{
		// Save size just before the user area
		*(int *)o2 = Size;
		o2 = (char *)o2 + sizeof(int);
	}

	if (o != NULL)
	{
		// Size is saved just before the user area
		OldSize = *(int *)((char *)o - sizeof(int));

		if (o2 != NULL)
			memcpy(o2, o, (int)Size < OldSize ? Size : OldSize);

		OldSize += sizeof(int);
		o = (char *)o - sizeof(int);
		FreeMem (o, OTHER, OldSize);
	}

	return(o2);
}


/*
==========
PrintMem
==========
*/
void PrintMem(void)
{
	char *rgszMemTypes[] = {"BSPEntity", "BSPPlane", "BSPTex", "BSPVertex", "BSPVis",
		"BSPNode", "BSPTexinfo", "BSPFace", "BSPLight", "BSPClipnode", "BSPLeaf",
		"BSPMarksurface", "BSPEdge", "BSPSurfedge", "BSPModel", "Mapface", "Mapbrush",
		"Mapentity","Winding", "Face", "Plane", "Portal", "Surface", "Node", "Brush",
		"Miptex", "World verts", "World edges", "Hash verts", "Other (KByte)",
		"Total (KByte)"};
	int i, Factor;

	if (options.fAllverbose || options.fMemverbose)
	{
		Message(msgLiteral, "\nData type     Current     Peak  KByte\n");
		for (i=0; i<=GLOBAL; i++)
		{
			Factor = i < GLOBAL - 1 ? 1 : 1024;

			Message(msgLiteral, "%-14s%7d %8d %6d\n", rgszMemTypes[i], rgMemActive[i] / Factor,
				rgMemPeak[i] / Factor, rgMemPeak[i] * rgcMemSize[i] / 1024);
		}
	}

	Message(msgLiteral, "\nPeak memory used : %.1f MB\n", (float)rgMemPeak[GLOBAL] / (1024 * 1024));
}

/*
============
FreeAllMem
============
*/
void FreeAllMem(void)
{
	int i, j;
	epair_t *ep, *next;

	for (i=0; i<map.cEntities; i++)
	{
		for (ep = map.rgEntities[i].epairs; ep; ep=next)
		{
			next = ep->next;
			if (ep->key)
				FreeMem(ep->key, OTHER, strlen(ep->key)+1);
			if (ep->value)
				FreeMem(ep->value, OTHER, strlen(ep->value)+1);
			FreeMem(ep, OTHER, sizeof(epair_t));
		}
		for (j=0; j<BSP_LUMPS; j++)
			if (map.rgEntities[i].pData[j])
				FreeMem(map.rgEntities[i].pData[j], j, map.rgEntities[i].cData[j]);
	}

	FreeMem(validfaces, OTHER, sizeof(face_t *)*cPlanes);
	FreeMem(pPlanes, PLANE, cPlanes);
	FreeMem(map.rgFaces, MAPFACE, map.cFaces);
	FreeMem(map.rgBrushes, MAPBRUSH, map.cBrushes);
	FreeMem(map.rgEntities, MAPENTITY, map.cEntities);

}


/*
=================
Message

Generic output of errors, warnings, stats, etc
=================
*/
void Message(int msgType, ...)
{
	va_list argptr;
	char szBuffer[512];
	char *szFmt;
	int ErrType;
	int Cur, Total;
	static bool fInPercent = false;

	va_start(argptr, msgType);

	// Exit if necessary
	if ((msgType == msgStat || msgType == msgProgress) && (!options.fVerbose || options.fNoverbose))
		return;
	else if (msgType == msgPercent && (options.fNopercent || options.fNoverbose))
		return;

	// Grab proper args
	if (msgType == msgWarning || msgType == msgError)
		ErrType = va_arg(argptr, int);
	else if (msgType == msgPercent)
	{
		Cur = va_arg(argptr, int);
		Total = va_arg(argptr, int);

		if (Total == -1)
		{
			fInPercent = false;
			ShowBar(-1, -1);
			return;
		}
	}
	else
		szFmt = va_arg(argptr, char *);

	// Handle warning/error-in-percent properly
	if (fInPercent && msgType != msgPercent)
		ShowBar(0, -1);

	switch (msgType)
	{
	case msgWarning:
		if (ErrType >= cWarnings)
		{
			printf("Internal error: Unknown ErrType (%d) in Message (msgWarning) !\n", ErrType);
			LogFile.Close();
			exit(1);
		}
		sprintf(szBuffer, "*** WARNING %02d: ", ErrType);
		vsprintf(szBuffer+strlen(szBuffer), rgszWarnings[ErrType], argptr);
		strcat(szBuffer, "\n");
		break;

	case msgError:
		if (ErrType >= cErrors)
			printf("Internal error: Unknown ErrType (%d) in Message (msgError) !\n", ErrType);
		else
		{
			sprintf(szBuffer, "*** ERROR %02d: ", ErrType);
			vsprintf(szBuffer+strlen(szBuffer), rgszErrors[ErrType], argptr);
			puts(szBuffer);
			LogFile.Printf("%s\n", szBuffer);
		}
		PrintFinish();
		LogFile.Close();
		exit(1);
		break;

	case msgLiteral:
		// Output as-is to screen and log file
		vsprintf(szBuffer, szFmt, argptr);
		break;

	case msgStat:
		// Output as-is to screen and log file
		strcpy(szBuffer, "       ");
		vsprintf(szBuffer+strlen(szBuffer), szFmt, argptr);		// Concatenate
		strcat(szBuffer, "\n");
		break;

	case msgProgress:
		// Output as-is to screen and log file
		strcpy(szBuffer, "---- ");
		vsprintf(szBuffer+strlen(szBuffer), szFmt, argptr);		// Concatenate
		strcat(szBuffer, " ----\n");
		break;

	case msgPercent:
		// Calculate the percent complete.  Only output if it changes.
		if (!fInPercent)
			ShowBar(0, 0);

		ShowBar(Cur, Total);

		// Handle output formatting properly
		fInPercent = true;
		return;

	case msgFile:
		// Output only to the file
		vsprintf(szBuffer, szFmt, argptr);
		break;

	case msgScreen:
		// Output only to the screen
		vsprintf(szBuffer, szFmt, argptr);
		break;

	default:
		printf("Unhandled msgType in message!\n");
		return;
	}

	if (msgType != msgFile)
	{
		printf("%s", szBuffer);
		fflush(stdout);
	}

	if (msgType != msgScreen)
	{
		LogFile.Printf("%s", szBuffer);
		LogFile.Flush();
	}

	va_end(argptr);
}
