// qbsp.c

#include "qbsp.h"
#include "wad.h"
#include "parser.h"

#define IntroString "TreeQBSP v1.81 -- Modified by Bengt Jardrup\n\n"

extern void SecToStr(time_t, char []);

double	  start = 0, end;

// command line flags
options_t options;

/*
===============
ProcessEntity
===============
*/
void ProcessEntity (void)
{
	surface_t	*surfs;
	node_t		*nodes;

	// No brushes means non-bmodel entity
	if (pCurEnt->iBrushStart == pCurEnt->iBrushEnd)
		return;

	if (map.iEntities > 0)
	{
		char mod[8];

		if (map.iEntities == 1)
			Message(msgProgress, "Internal Entities");
		sprintf (mod, "*%i", map.cTotal[BSPMODEL]);
		if (options.fVerbose)
			PrintEntity (map.iEntities);

		if (hullnum == 0)
			Message(msgStat, "MODEL: %s", mod);
		SetKeyValue (map.iEntities, "model", mod);
	}

	// take the brush_ts and clip off all overlapping and contained faces,
	// leaving a perfect skin of the model with no hidden faces
	Brush_LoadEntity();

	if (!pCurEnt->pBrushes)
	{
		options.fVerbose = true;
		PrintEntity (map.iEntities);
		Message(msgError, errNoValidBrushes);
	}

	surfs = CSGFaces();

	FreeBrushsetBrushes();

	if (hullnum != 0)
	{
		nodes = SolidBSP (surfs, true);
		if (map.iEntities == 0 && !options.fNofill)	// assume non-world bmodels are simple
		{
			PortalizeWorld (nodes, false);
			if (FillOutside (nodes))
			{
				// Free portals before regenerating new nodes
				FreeAllPortals (nodes);
				surfs = GatherNodeFaces (nodes);
				nodes = SolidBSP (surfs, false);	// make a really good tree
			}
		}
		ExportNodePlanes(nodes);
		ExportClipNodes(nodes);
	}
	else
	{
		// SolidBSP generates a node tree
		//
		// if not the world, make a good tree first
		// the world is just going to make a bad tree
		// because the outside filling will force a regeneration later
		nodes = SolidBSP (surfs, map.iEntities == 0);

		// build all the portals in the bsp tree
		// some portals are solid polygons, and some are paths to other leafs
		if (map.iEntities == 0 && !options.fNofill)	// assume non-world bmodels are simple
		{
			PortalizeWorld (nodes, false);

			if (FillOutside (nodes))
			{
				FreeAllPortals (nodes);

				// get the remaining faces together into surfaces again
				surfs = GatherNodeFaces (nodes);

				// merge polygons
				MergeAll (surfs);

				// make a really good tree
				nodes = SolidBSP (surfs, false);

				// make the real portals for vis tracing
				PortalizeWorld (nodes, true);

				// fix tjunctions
				if (options.fTjunc)
					tjunc (nodes);
			}
			FreeAllPortals (nodes);
		}

		ExportNodePlanes (nodes);
		MakeFaceEdges (nodes);
		ExportDrawNodes (nodes);
	}

	map.cTotal[BSPMODEL]++;
}

/*
=================
UpdateEntLump

=================
*/
void UpdateEntLump (void)
{
	int m, iEntity;
	char szMod[80];
	char *szClassname;
	vec3_t temp;

	Message(msgStat, "Updating entities lump...");

	VectorCopy(vec3_origin, temp);
	m = 1;
	for (iEntity = 1; iEntity < map.cEntities; iEntity++)
	{
		if (map.rgEntities[iEntity].iBrushStart == map.rgEntities[iEntity].iBrushEnd)
			continue;
		sprintf (szMod, "*%i", m);
		SetKeyValue (iEntity, "model", szMod);
		m++;

		// Do extra work for rotating entities if necessary
		szClassname = ValueForKey(iEntity, "classname");
		if (!strncmp(szClassname, "rotate_", 7))
			FixRotateOrigin(iEntity, temp);
	}

	LoadBSPFile();
	WriteEntitiesToString();
	WriteBSPFile();

	if (!options.fAllverbose)
		options.fVerbose = false;
}


/*
=================
CreateSingleHull

=================
*/
void CreateSingleHull (void)
{
	Message(msgLiteral, "Processing hull %d...\n", hullnum);
	map.cTotal[BSPMODEL] = 0;

	// for each entity in the map file that has geometry
	for (map.iEntities=0, pCurEnt = &map.rgEntities[0];
		 map.iEntities < map.cEntities;
		 map.iEntities++, pCurEnt++)
	{
		ProcessEntity();
		if (!options.fAllverbose)
			options.fVerbose = false;	// don't print rest of entities
	}
}

/*
=================
CreateHulls

=================
*/
void CreateHulls (void)
{
	// create the hulls sequentially
	hullnum = 0;
	CreateSingleHull ();

	hullnum = 1;
	CreateSingleHull ();

	hullnum = 2;
	CreateSingleHull ();
}


/*
=================
ProcessFile
=================
*/
void ProcessFile (void)
{
	char *szWad, WadName[MAXTOKEN + 1];
	WAD wad;

	// load brushes and entities
	LoadMapFile();
	if (options.fOnlyents)
	{
		UpdateEntLump ();
		return;
	}

	szWad = ValueForKey(0, "_wad");

	if (!szWad || !szWad[0])
		szWad = ValueForKey(0, "wad");

	if (!szWad || !szWad[0])
	{
		Message(msgWarning, warnNoWadKey);
		WadName[0] = '\0';
	}
	else
		strcpy(WadName, szWad);

	if (!wad.InitWADList(WadName))
	{
		Message(msgWarning, warnNoValidWads);
		pWorldEnt->cTexdata = 0;
	}

	// init the tables to be shared by all models
	BeginBSPFile ();

	if (options.fNoverbose)
		options.fVerbose = false;

	// the clipping hulls will not be written out to text files by forked processes :)
	CreateHulls ();

	WriteEntitiesToString();
	wad.fProcessWAD();
	FinishBSPFile ();
}


/*
==============
PrintOptions
==============
*/
void PrintOptions(void)
{
	Message(msgLiteral, "TreeQBSP performs geometric level processing of Quake .MAP files to create\n");
	Message(msgLiteral, "Quake .BSP files.\n\n");
	Message(msgLiteral, "treeqbsp [options] sourcefile [destfile]\n\n");
	Message(msgLiteral, "Options:\n");
	Message(msgLiteral, "   -notjunc        Disable tjunc calculations\n");
	Message(msgLiteral, "   -nofill         Disable outside filling\n");
	Message(msgLiteral, "   -fill           Force outside filling in hulls 1/2 if they leak\n");
	Message(msgLiteral, "   -onlyents       Only update .MAP entities\n");
	Message(msgLiteral, "   -verbose        Print out more .MAP information\n");
	Message(msgLiteral, "   -verbosemem     Print out more memory information\n");
	Message(msgLiteral, "   -noverbose      Print out almost no information at all\n");
	Message(msgLiteral, "   -splitspecial   Disable merging of sky and water faces\n");
	Message(msgLiteral, "   -transwater     Compute portal information for transparent water\n");
	Message(msgLiteral, "   -transsky       Compute portal information for transparent sky\n");
	Message(msgLiteral, "   -oldaxis        Use original QBSP texture alignment algorithm\n");
	Message(msgLiteral, "   -oldtexpos      Disable support for enhanced texture positioning\n");
	Message(msgLiteral, "   -bspleak        Create a .POR file, used in the BSP editor\n");
	Message(msgLiteral, "   -oldleak        Create an old-style QBSP .PTS file\n");
	Message(msgLiteral, "   -nosortface     Disable face sorting before processing\n");
	Message(msgLiteral, "   -nopercent      Disable percent completion information\n");
	Message(msgLiteral, "   -leak [n]       Leak # to generate leakfile for (default 1)\n");
	Message(msgLiteral, "   -leakdist [n]   Distance between leakfile points (default 4)\n");
	Message(msgLiteral, "   -simpdist [n]   Simplification distance in leakfile (default 100)\n");
	Message(msgLiteral, "   -subdivide [n]  Use different texture subdivision (default 240)\n");
	Message(msgLiteral, "   sourcefile      .MAP file to process\n");
	Message(msgLiteral, "   destfile        .BSP file to output\n");

	LogFile.Close();

	exit(1);
}


/*
=============
GetTok

Gets tokens from command line string.
=============
*/
char *GetTok(char *szBuf, char *szEnd)
{
	char *szTok;

	if (szBuf >= szEnd)
		return NULL;

	// Eliminate leading whitespace
	while (*szBuf == ' ' || *szBuf == '\n' || *szBuf == '\t' ||
		   *szBuf == '\r')
		szBuf++;

	if (szBuf >= szEnd)
		return NULL;

	// Three cases: strings, options, and none-of-the-above.
	if (*szBuf == '\"')
	{
		szBuf++;
		szTok = szBuf;
		while (*szBuf != 0 && *szBuf != '\"' && *szBuf != '\n' && *szBuf != '\r')
			szBuf++;
	}
	else if (*szBuf == '-' || *szBuf == '/')
	{
		szTok = szBuf;
		while (*szBuf != ' ' && *szBuf != '\n' && *szBuf != '\t' &&
			   *szBuf != '\r' && *szBuf != 0)
			szBuf++;
	}
	else
	{
		szTok = szBuf;
		while (*szBuf != ' ' && *szBuf != '\n' && *szBuf != '\t' &&
			   *szBuf != '\r' && *szBuf != 0)
			szBuf++;
	}

	if (*szBuf != 0)
		*szBuf = 0;
	return szTok;
}

/*
==================
ParseOptions
==================
*/
void ParseOptions(char *szOptions)
{
	char *szTok, *szTok2;
	char *szEnd;
	int NameCount = 0;

	szEnd = szOptions + strlen(szOptions);
	szTok = GetTok(szOptions, szEnd);
	while (szTok)
	{
		if (szTok[0] != '-' && szTok[0] != '/')
		{
			// Treat as filename
			if (NameCount == 0)
				strcpy(options.szMapName, szTok);
			else if (NameCount == 1)
				strcpy(options.szBSPName, szTok);
			else
				Message(msgError, errUnknownOption, szTok);
			NameCount++;
		}
		else
		{
			szTok++;
			if (!stricmp (szTok, "tjunc"))
				;
			else if (!stricmp (szTok, "notjunc"))
				options.fTjunc = false;
			else if (!stricmp (szTok, "nofill"))
				options.fNofill = true;
			else if (!stricmp (szTok, "fill"))
				options.fForcedFill = true;
			else if (!stricmp (szTok, "onlyents"))
				options.fOnlyents = true;
			else if (!stricmp (szTok, "verbose"))
				options.fAllverbose = true;
			else if (!stricmp (szTok, "verbosemem"))
				options.fMemverbose = true;
			else if (!stricmp (szTok, "splitspecial"))
				options.fSplitspecial = true;
			else if (!stricmp (szTok, "transwater"))
				options.fTranswater = true;
			else if (!stricmp (szTok, "transsky"))
				options.fTranssky = true;
			else if (!stricmp (szTok, "oldaxis"))
				options.fOldaxis = true;
			else if (!stricmp (szTok, "oldtexpos"))
				options.fOldtexpos = true;
			else if (!stricmp (szTok, "bspleak"))
				options.fBspleak = true;
			else if (!stricmp (szTok, "noverbose"))
				options.fNoverbose = true;
			else if (!stricmp (szTok, "nosortface"))
				options.fSortFace = false;
			else if (!stricmp (szTok, "oldleak"))
				options.fOldleak = true;
			else if (!stricmp (szTok, "nopercent"))
				options.fNopercent = true;
			else if (!stricmp (szTok, "percent"))
				;
			else if (!stricmp (szTok, "leak"))
			{
				szTok2 = GetTok(szTok+strlen(szTok)+1, szEnd);
				if (!szTok2)
					Message(msgError, errInvalidOption, szTok);
				options.dxLeakNo = atoi(szTok2) - 1;
				szTok = szTok2;
			}
			else if (!stricmp (szTok, "leakdist"))
			{
				szTok2 = GetTok(szTok+strlen(szTok)+1, szEnd);
				if (!szTok2)
					Message(msgError, errInvalidOption, szTok);
				options.dxLeakDist = atoi(szTok2);

				if (options.dxLeakDist < 1)
					Message(msgError, errInvalidLeakDist, szTok);

				szTok = szTok2;
			}
			else if (!stricmp (szTok, "simpdist"))
			{
				szTok2 = GetTok(szTok+strlen(szTok)+1, szEnd);
				if (!szTok2)
					Message(msgError, errInvalidOption, szTok);
				options.dxSimpDist = atoi(szTok2);

				if (options.dxSimpDist < 2)
					Message(msgError, errInvalidSimpDist, szTok);

				szTok = szTok2;
			}
			else if (!stricmp (szTok, "subdivide"))
			{
				szTok2 = GetTok(szTok+strlen(szTok)+1, szEnd);
				if (!szTok2)
					Message(msgError, errInvalidOption, szTok);
				options.dxSubdivide = atoi(szTok2);
				szTok = szTok2;
			}
			else if (!stricmp(szTok, "?") ||
					 !stricmp(szTok, "help"))
				PrintOptions();
			else
				Message(msgError, errUnknownOption, szTok);
		}

		szTok = GetTok(szTok+strlen(szTok)+1, szEnd);
	}
}

/*
==================
RemoveFile
==================
*/
static void RemoveFile(char Ext[])
{
	StripExtension(options.szBSPName);
	strcat(options.szBSPName, Ext);
	remove(options.szBSPName);
}

/*
==================
InitQBSP
==================
*/
void InitQBSP(int argc, char **argv)
{
	int i;
	char szArgs[512];
	char *szBuf;
	int length;
	File IniFile;

	// Start logging to qbsp.log
	if (!LogFile.fOpen("qbsp.log", "wt", false))
		Message(msgWarning, warnNoLogFile);
	else
		// Kinda dumb, but hey...
		Message(msgFile, IntroString);

	// Initial values
	options.dxLeakDist = 4;
	options.dxSimpDist = 100;
	options.dxSubdivide = 240;
	options.fSortFace = true;
	options.fTjunc = true;
	options.fVerbose = true;
	options.szMapName[0] = options.szBSPName[0] = 0;

	length = IniFile.LoadFile("qbsp.ini", (void **)&szBuf, false);
	if (length)
	{
		Message(msgLiteral, "Loading options from qbsp.ini\n");
		ParseOptions(szBuf);

		FreeMem(szBuf, OTHER, length+1);
	}

	// Concatenate command line args
	szArgs[0] = 0;
	for (i=1; i<argc; i++)
	{
		// Toss " around filenames to preserve LFNs in the Parsing function
		if (argv[i][0] != '-')
			strcat(szArgs, "\"");
		strcat(szArgs, argv[i]);
		if (argv[i][0] != '-')
			strcat(szArgs, "\" ");
		else
			strcat(szArgs, " ");
	}

	ParseOptions(szArgs);

	if (options.szMapName[0] == 0)
		PrintOptions();

	// create destination name if not specified
	DefaultExtension (options.szMapName, ".map");

	if (!stricmp(strrchr(options.szMapName, '.'), ".bsp"))
		Message(msgError, errNoBSPExtension, options.szMapName);

	// The .map extension gets removed right away anyways...
	if (options.szBSPName[0] == 0)
		strcpy(options.szBSPName, options.szMapName);

	// Remove already existing files
	if (!options.fOnlyents)
	{
		RemoveFile(".bsp");

		// Probably not the best place to do this
		Message(msgLiteral, "Input file: %s\n", options.szMapName);
		Message(msgLiteral, "Output file: %s\n\n", options.szBSPName);

		RemoveFile(".prt");
		RemoveFile(".pts");
		RemoveFile(".por");
		RemoveFile(".h1");
		RemoveFile(".h2");
	}
}

/*
==================
SecToStr
==================
*/
static void SecToStr(time_t Sec, char Str[])

{
	int Hour;

	Hour = Sec / 3600;

	if (Hour == 0)
		sprintf(Str, "%d:%02d", (Sec / 60) % 60, Sec % 60);
	else
		sprintf(Str, "%d:%02d:%02d", Hour, (Sec / 60) % 60, Sec % 60);
}

/*
==================
PrintFinish
==================
*/
void PrintFinish(void)
{
	char Str[20];

	if (start == 0)
		return;

	end = I_FloatTime ();

	SecToStr((time_t)(end - start + 0.5), Str);

	Message(msgLiteral, "\nElapsed time : %s\n", Str);

	PrintMem();
}

/*
==================
main
==================
*/
int main (int argc, char **argv)
{
	Message(msgScreen, IntroString);

	InitQBSP(argc, argv);

	// do it!
	start = I_FloatTime ();
	ProcessFile();

	PrintFinish();

//	FreeAllMem();
//	PrintMem();

	LogFile.Close();

	return 0;
}