// map.c

#include "qbsp.h"
#include "parser.h"

#define info_player_start		1
#define info_player_deathmatch	2
#define	info_player_coop		4

int cAnimtex;
int rgfStartSpots;

void CleanupName (char *in, char *out)
{
	int		i;
	
	for (i = 0; i < sizeof(miptex_t) - 1; i++)
	{
		if (!in[i])
			break;
			
		out[i] = tolower(in[i]);
	}
	
	for (; i < sizeof(miptex_t); i++)
		out[i] = 0;
}

/*
===============
FindMiptex
===============
*/
int FindMiptex (char *szName)
{
	int i;

	for (i=0 ; i<cMiptex; i++)
	{
		if (!stricmp (szName, rgszMiptex[i]))
			return i;
	}
	CleanupName (szName, rgszMiptex[i]);
	cMiptex++;

	if (szName[0] == '+')
		cAnimtex++;

	return i;
}

/*
===============
FindTexinfo

Returns a global texinfo number
===============
*/
int	FindTexinfo (texinfo_t *t)
{
	int			i, j;
	texinfo_t	*tex;
		
	// set the special flag
	if ((rgszMiptex[t->miptex][0] == '*' || !strnicmp(rgszMiptex[t->miptex], "sky",3)) && !options.fSplitspecial)
		t->flags |= TEX_SPECIAL;

	tex = pWorldEnt->pTexinfo;
	for (i=0; i<pWorldEnt->iTexinfo; i++, tex++)
	{
		if (t->miptex != tex->miptex)
			continue;
		if (t->flags != tex->flags)
			continue;
		
		for (j=0 ; j<8 ; j++)
			if (t->vecs[0][j] != tex->vecs[0][j])
				break;
		if (j != 8)
			continue;
			
		return i;
	}
	
	// allocate a new texture
	pWorldEnt->pTexinfo[i] = *t;
	pWorldEnt->iTexinfo++;
	map.cTotal[BSPTEXINFO]++;

	return i;
}


//============================================================================

/*
=================
ParseEpair
=================
*/
void ParseEpair (Parser *pParser)
{
	epair_t	*e;
	char *szToken;
	
	e = (epair_t *)AllocMem(OTHER, sizeof(epair_t));
	e->next = map.rgEntities[map.iEntities].epairs;
	map.rgEntities[map.iEntities].epairs = e;
	
	szToken = pParser->GetToken();

	if (strlen(szToken) >= MAX_KEY-1)
		Message(msgError, errEpairTooLong, pParser->GetLineNum());
	e->key = copystring(szToken);
	pParser->ParseToken (false);
	if (strlen(szToken) >= MAX_VALUE-1)
		Message(msgError, errEpairTooLong, pParser->GetLineNum());
	e->value = copystring(szToken);

	if (!stricmp(e->key, "origin"))
		GetVectorForKey(map.iEntities, e->key, map.rgEntities[map.iEntities].origin);
	else if (!stricmp(e->key, "classname"))
	{
		if (!stricmp(e->value, "info_player_start"))
		{
			if (rgfStartSpots & info_player_start)
				Message(msgWarning, warnMultipleStarts);
			rgfStartSpots |= info_player_start;
		}
		else if (!stricmp(e->value, "info_player_deathmatch"))
			rgfStartSpots |= info_player_deathmatch;
		else if (!stricmp(e->value, "info_player_coop"))
			rgfStartSpots |= info_player_coop;
	}
}

//============================================================================


/*
==================
textureAxisFromPlane
==================
*/
void TextureAxisFromPlane(plane_t *pln, vec3_t xv, vec3_t yv)
{
	vec3_t	baseaxis[18] =
	{
		{0,0,1}, {1,0,0}, {0,-1,0},			// floor
		{0,0,-1}, {1,0,0}, {0,-1,0},		// ceiling
		{1,0,0}, {0,1,0}, {0,0,-1},			// west wall
		{-1,0,0}, {0,1,0}, {0,0,-1},		// east wall
		{0,1,0}, {1,0,0}, {0,0,-1},			// south wall
		{0,-1,0}, {1,0,0}, {0,0,-1}			// north wall
	};

	int		bestaxis;
	vec_t	dot,best;
	int		i;
	
	best = 0;
	bestaxis = 0;
	
	for (i=0 ; i<6 ; i++)
	{
		dot = DotProduct (pln->normal, baseaxis[i*3]);
		if (dot > best ||
		   (dot == best && !options.fOldaxis))
		{
			best = dot;
			bestaxis = i;
		}
	}
	
	VectorCopy (baseaxis[bestaxis*3+1], xv);
	VectorCopy (baseaxis[bestaxis*3+2], yv);
}


//=============================================================================

#define ScaleCorrection	(1.0/128.0)

static int Weight2(vec_t n)
{
	if (fabs(n - 1) < 0.9)
		return(0);

	if (fabs(n + 1) < 0.9)
		return(1);

	return(2);
}

static int Weight(vec3_t n)
{
	return(Weight2(n[0]) * 100 + Weight2(n[1]) * 10 + Weight2(n[2]));
}

static bool Phase1;

static int CmpFace(const void *arg1, const void *arg2)
{
	mapface_t *f1, *f2;
	int	  Cmp;

	f1 = (mapface_t *)arg1;
	f2 = (mapface_t *)arg2;

	Cmp = Weight(f1->plane.normal) - Weight(f2->plane.normal);

	if (Phase1)
		Cmp += (int)(1000 * (f1->plane.dist - f2->plane.dist)); // To get determinism, we need this

	return(Cmp);
}

static void SortFaces(void)
{
	int Faces;

	Faces = map.rgBrushes[map.iBrushes].iFaceEnd - (map.iFaces + 1);

	if (Faces < 2)
		return; // Too few faces

	Phase1 = true;

	// 1st sort in a deterministic order
	qsort(&map.rgFaces[map.iFaces + 1], Faces, sizeof(mapface_t), CmpFace);

	Phase1 = false;

	// 2nd sort in a "neat" order
	qsort(&map.rgFaces[map.iFaces + 1], Faces, sizeof(mapface_t), CmpFace);
}

/*
=================
ParseBrush
=================
*/
void ParseBrush (Parser *pParser)
{
	vec3_t		planepts[3];
	vec3_t		t1, t2, t3;
	int			i,j;
	texinfo_t	tx;
	mapbrush_t	*Brush;
	mapface_t	*Face;
	vec_t		d;
	int			shift[2], rotate;
	vec_t		scale[2];
	char *szToken;
	int iFace;
	bool		ok;

	Brush = &map.rgBrushes[map.iBrushes];
	Brush->iFaceEnd = map.iFaces+1;
	Brush->iLine = pParser->GetLineNum();
	
	szToken = pParser->GetToken();

	ok = pParser->ParseToken (true);

	while (ok)
	{
		pParser->txcommand = 0;
		if (!strcmp (szToken, "}") )
		{
			if (!options.fOnlyents)
			{
				if (map.rgBrushes[map.iBrushes].iFaceEnd - (map.iFaces + 1) < 4)
					Message(msgError, errTooFewFaces, Brush->iLine); // Too few faces in brush
			}

			if (options.fSortFace)
				SortFaces();

			break;
		}
		
		// read the three point plane definition
		for (i=0; i<3; i++)
		{
			if (i != 0)
				pParser->ParseToken (true);
			if (strcmp (szToken, "(") )
				Message(msgError, errInvalidMapPlane, pParser->GetLineNum());
			
			for (j=0 ; j<3 ; j++)
			{
				pParser->ParseToken (false);
				planepts[i][j] = atof(szToken);
			}
			
			pParser->ParseToken (false);
			if (strcmp (szToken, ")") )
				Message(msgError, errInvalidMapPlane, pParser->GetLineNum());
				
		}

		// read the texturedef
		memset (&tx, 0, sizeof(tx));
		pParser->ParseToken (false);

		// Check texture name length
		if (strlen(szToken) > sizeof(miptex_t) - 1)
			Message(msgError, errInvalidTexture, pParser->GetLineNum(), szToken);

		tx.miptex = FindMiptex (szToken);
		pParser->ParseToken (false);
		shift[0] = atoi(szToken);
		pParser->ParseToken (false);
		shift[1] = atoi(szToken);
		pParser->ParseToken (false);
		rotate = atoi(szToken);	
		pParser->ParseToken (false);
		scale[0] = atof(szToken);
		pParser->ParseToken (false);
		scale[1] = atof(szToken);

		ok = pParser->ParseToken (true); // Note : line number normally gets advanced here

		// if the three points are all on a previous plane, it is a
		// duplicate plane
		for (iFace = Brush->iFaceEnd-1; iFace > map.iFaces; iFace--)
		{
			for (i=0; i<3; i++)
			{
				d = DotProduct(planepts[i], map.rgFaces[iFace].plane.normal) - 
					map.rgFaces[iFace].plane.dist;
				if (d < -ON_EPSILON || d > ON_EPSILON)
					break;
			}
			if (i==3)
				break;
		}
		if (iFace > map.iFaces)
		{
			Message(msgWarning, warnBrushDuplicatePlane, pParser->GetLineNum() - 1);
			continue;
		}

		if (map.iFaces < 0)
			Message(msgError, errLowFaceCount);
		
		Face = &map.rgFaces[map.iFaces];
		
		// convert to a vector / dist plane
		for (j=0; j<3; j++)
		{
			t1[j] = planepts[0][j] - planepts[1][j];
			t2[j] = planepts[2][j] - planepts[1][j];
			t3[j] = planepts[1][j];
		}
		
		CrossProduct(t1,t2, Face->plane.normal);
		if (VectorCompare (Face->plane.normal, vec3_origin))
		{
			Message(msgWarning, warnNoPlaneNormal, pParser->GetLineNum() - 1);
			continue;
		}
		VectorNormalize (Face->plane.normal);
		Face->plane.dist = DotProduct (t3, Face->plane.normal);

		if (pParser->txcommand == '1' || pParser->txcommand == '2')
		{		// from QuArK, the texture vectors are given directly from the three points
			vec3_t	TexPt[2];
			int		k;
			vec_t	dot22, dot23, dot33, mdet, aa, bb, dd;
			
			k = pParser->txcommand-'0';
			for (j=0; j<3; j++)
				TexPt[1][j] = (planepts[k][j] - planepts[0][j]) * ScaleCorrection;
			k = 3-k;
			for (j=0; j<3; j++)
				TexPt[0][j] = (planepts[k][j] - planepts[0][j]) * ScaleCorrection;

			dot22 = DotProduct (TexPt[0], TexPt[0]);
			dot23 = DotProduct (TexPt[0], TexPt[1]);
			dot33 = DotProduct (TexPt[1], TexPt[1]);
			mdet = dot22*dot33-dot23*dot23;

			if (mdet<1E-6 && mdet>-1E-6)
			{
				aa = bb = dd = 0;
				Message(msgWarning, warnDegenerateQuArK, pParser->GetLineNum() - 1);
			}
			else
			{
				mdet = 1.0/mdet;
      				aa = dot33*mdet;
      				bb = -dot23*mdet;
				//cc = -dot23*mdet;     // cc = bb
				dd = dot22*mdet;
			}

			for (j=0; j<3; j++)
			{
				tx.vecs[0][j] = aa * TexPt[0][j] + bb * TexPt[1][j];
				tx.vecs[1][j] = -(/*cc*/ bb * TexPt[0][j] + dd * TexPt[1][j]);
			}
			
			tx.vecs[0][3] = -DotProduct(tx.vecs[0], planepts[0]);
			tx.vecs[1][3] = -DotProduct(tx.vecs[1], planepts[0]);
		}
		else
		// fake proper texture vectors from QuakeEd style
		{
			vec3_t	vecs[2];
			int		sv, tv;
			vec_t	ang, sinv, cosv;
			vec_t	ns, nt;
			
			TextureAxisFromPlane(&(Face->plane), vecs[0], vecs[1]);
		
			if (!scale[0])
				scale[0] = 1;
			if (!scale[1])
				scale[1] = 1;
		
		
			// rotate axis
			if (rotate == 0)
				{ sinv = 0 ; cosv = 1; }
			else if (rotate == 90)
				{ sinv = 1 ; cosv = 0; }
			else if (rotate == 180)
				{ sinv = 0 ; cosv = -1; }
			else if (rotate == 270)
				{ sinv = -1 ; cosv = 0; }
			else
			{	
				ang = (vec_t)rotate / 180 * Q_PI;
				sinv = sin(ang);
				cosv = cos(ang);
			}
		
			if (vecs[0][0])
				sv = 0;
			else if (vecs[0][1])
				sv = 1;
			else
				sv = 2;
						
			if (vecs[1][0])
				tv = 0;
			else if (vecs[1][1])
				tv = 1;
			else
				tv = 2;
							
			for (i=0 ; i<2 ; i++)
			{
				ns = cosv * vecs[i][sv] - sinv * vecs[i][tv];
				nt = sinv * vecs[i][sv] +  cosv * vecs[i][tv];
				vecs[i][sv] = ns;
				vecs[i][tv] = nt;
			}
		
			for (i=0 ; i<2 ; i++)
				for (j=0 ; j<3 ; j++)
					tx.vecs[i][j] = vecs[i][j] / scale[i];
		
			tx.vecs[0][3] = (vec_t)shift[0];
			tx.vecs[1][3] = (vec_t)shift[1];
		}
	
		// unique the texinfo
		Face->texinfo = FindTexinfo (&tx);
		map.iFaces--;
	}

	Brush->iFaceStart = map.iFaces+1;
	map.iBrushes--;
}

/*
================
ParseEntity
================
*/
bool ParseEntity (Parser *pParser)
{
	char *szToken, *szClassname;

	szToken = pParser->GetToken();

	if (!pParser->ParseToken(true))
		return false;

	if (strcmp (szToken, "{") )
		Message(msgError, errParseEntity, pParser->GetLineNum());
	
	if (map.iEntities >= map.cEntities)
		Message(msgError, errLowEntCount);

	pCurEnt->iBrushEnd = map.iBrushes+1;
	
	do
	{
		if (!pParser->ParseToken(true))
			Message(msgError, errUnexpectedEOF);
		if (!strcmp (szToken, "}"))
			break;
		else if (!strcmp (szToken, "{") )
			ParseBrush (pParser);
		else
			ParseEpair (pParser);
	} while (1);

	szClassname = ValueForKey(map.iEntities, "classname");

	if (!szClassname)
		Message(msgError, errNoClassname, pParser->GetLineNum()); // Missing classname

	// Allocate some model memory while we're here
	pCurEnt->iBrushStart = map.iBrushes+1;
	if (pCurEnt->iBrushStart != pCurEnt->iBrushEnd)
	{
		pCurEnt->pModels = (dmodel_t *)AllocMem(BSPMODEL, 1);
		pCurEnt->cModels = 1;
	}
	else if (!options.fOnlyents)
	{
		if (!stricmp(szClassname, "worldspawn"))
			Message(msgError, errNoWorldBrushes, pParser->GetLineNum()); // No world brushes
	}

	map.iEntities++;
	pCurEnt++;

	return true;
}

/*
================
PreParseFile
================
*/
void PreParseFile (char *buf)
{
	int braces = 0;

	map.cEntities = map.cBrushes = map.cFaces = 0;

	// Very simple... we just want numbers here.  Invalid formats are
	// detected later.  Problems with deviant .MAP formats.
	while (*buf != 0)
	{
		if (*buf == '\"')
		{
			buf++;
			// Quoted string... skip to end of quote
			while (*buf != '\"' && *buf)
				buf++;
			if (!*buf)
				break;
		}
		else if (*buf == '/' && *(buf+1) == '/')
		{
			// Comment... skip to end of line
			while (*buf != '\n' && *buf)
				buf++;
			if (!*buf)
				break;
		}
		else if (*buf == '{')
		{
			if (braces == 0)
				map.cEntities++;
			else if (braces == 1)
				map.cBrushes++;
			braces++;
		}
		else if (*buf == '}')
			braces--;
		else if (*buf == '(')
			map.cFaces++;
		buf++;
	}

	if (map.cFaces % 3 != 0)
		Message(msgWarning, warnBadMapFaceCount);
	map.cFaces /= 3;

	map.rgFaces = (mapface_t *)AllocMem(MAPFACE, map.cFaces);
	map.rgBrushes = (mapbrush_t *)AllocMem(MAPBRUSH, map.cBrushes);
	map.rgEntities = (mapentity_t *)AllocMem(MAPENTITY, map.cEntities);

	// While we're here...
	pWorldEnt = map.rgEntities;

	// Allocate maximum memory here, copy over later
	// Maximum possible is one miptex/texinfo per face
	rgszMiptex = (miptex_t *)AllocMem(MIPTEX, map.cFaces);
	pWorldEnt->pTexinfo = (texinfo_t *)AllocMem(BSPTEXINFO, map.cFaces);
	pWorldEnt->cTexinfo = map.cFaces;
}

/*
================
LoadMapFile
================
*/
void LoadMapFile(void)
{
	Parser  *pParser;
	File	MapFile;
	plane_t *UPlane;
	char	*pBuf;
	int	i, j, k, length;
	void	*pTemp;
	bool	fUnique;
		
	Message(msgProgress, "LoadMapFile");

	length = MapFile.LoadFile(options.szMapName, (void **)&pBuf);
	PreParseFile(pBuf);

	pParser = new Parser(pBuf);

	// Faces are loaded in reverse order, to be compatible with origqbsp.
	// Brushes too.
	map.iFaces = map.cFaces-1;
	map.iBrushes = map.cBrushes-1;
	map.iEntities = 0;
	pCurEnt = &map.rgEntities[0];
	
	while (ParseEntity (pParser))
		;
	
	FreeMem(pBuf, OTHER, length+1);
	delete pParser;

	// Print out warnings for entities
	if (!(rgfStartSpots & info_player_start))
		Message(msgWarning, warnNoPlayerStart);
	if (!(rgfStartSpots & info_player_deathmatch))
		Message(msgWarning, warnNoPlayerDeathmatch);
//	if (!(rgfStartSpots & info_player_coop))
//		Message(msgWarning, warnNoPlayerCoop);

	// Clean up texture memory
	if (cMiptex > map.cFaces)
		Message(msgError, errLowMiptexCount);
	else if (cMiptex < map.cFaces)
	{
		// For stuff in AddAnimatingTex, make room available
		pTemp = (void *)rgszMiptex;
		rgszMiptex = (miptex_t *)AllocMem(MIPTEX, cMiptex + cAnimtex * 20);
		memcpy(rgszMiptex, pTemp, cMiptex * rgcMemSize[MIPTEX]);
		FreeMem(pTemp, MIPTEX, map.cFaces);
	}

	if (pWorldEnt->iTexinfo > pWorldEnt->cTexinfo)
		Message(msgError, errLowTexinfoCount);
	else if (pWorldEnt->iTexinfo < pWorldEnt->cTexinfo)
	{
		pTemp = (void *)pWorldEnt->pTexinfo;
		pWorldEnt->pTexinfo = (texinfo_t *)AllocMem(BSPTEXINFO, pWorldEnt->iTexinfo);
		memcpy(pWorldEnt->pTexinfo, pTemp, pWorldEnt->iTexinfo * rgcMemSize[BSPTEXINFO]);
		FreeMem(pTemp, BSPTEXINFO, pWorldEnt->cTexinfo);
		pWorldEnt->cTexinfo = pWorldEnt->iTexinfo;
	}

	// Allocate temporary memory for unique planes
	UPlane = (plane_t *)AllocMem(PLANE, map.cFaces);

	// One plane per face + 6 for portals
	cPlanes = map.cFaces + 6;

	// Count # of unique planes
	for (i=k=0; i<map.cFaces; i++)
	{
		fUnique = true;
		for (j=0; j<k; j++)
		{
			if (fabs(map.rgFaces[i].plane.dist - UPlane[j].dist) < EQUAL_EPSILON &&
			    VectorCompare(map.rgFaces[i].plane.normal, UPlane[j].normal))
			{
				fUnique = false;
				cPlanes--;
				break;
			}
		}

		if (fUnique)
			UPlane[k++] = map.rgFaces[i].plane;

		Message(msgPercent, i, map.cFaces);
	}

	Message(msgPercent, -1, -1);

	FreeMem(UPlane, PLANE, map.cFaces);

	// Now iterate through brushes, add one plane for each face below 6 axis aligned faces.
	// This compensates for planes added in ExpandBrush.
	int cAxis;
	for (i=0; i<map.cBrushes; i++)
	{
		cAxis = 0;
		for (j=map.rgBrushes[i].iFaceStart; j<map.rgBrushes[i].iFaceEnd; j++)
		{
			if ((fabs(map.rgFaces[j].plane.normal[0]) - (1-0.0001)) > 0 ||
				(fabs(map.rgFaces[j].plane.normal[1]) - (1-0.0001)) > 0 ||
				(fabs(map.rgFaces[j].plane.normal[2]) - (1-0.0001)) > 0)
				cAxis++;
		}
		if (6-cAxis > 0)
			cPlanes += 6-cAxis;
	}

	// cPlanes*3 because of 3 hulls, then add 20% as a fudge factor for hull edge bevel planes
	cPlanes = 3*cPlanes + cPlanes/5;
	pPlanes = (plane_t *)AllocMem(PLANE, cPlanes);

	Message(msgStat, "%6i faces", map.cFaces);
	Message(msgStat, "%6i brushes", map.cBrushes);
	Message(msgStat, "%6i entities", map.cEntities);
	Message(msgStat, "%6i unique texnames", cMiptex);
	Message(msgStat, "%6i texinfo", pWorldEnt->cTexinfo);
	Message(msgLiteral, "\n");
}

void PrintEntity(int iEntity)
{
	epair_t	*ep;
	
	for (ep=map.rgEntities[iEntity].epairs ; ep ; ep=ep->next)
		Message(msgStat, "%20s : %s", ep->key, ep->value);
}


char *ValueForKey(int iEntity, char *key)
{
	epair_t	*ep;
	
	for (ep=map.rgEntities[iEntity].epairs; ep; ep=ep->next)
		if (!strcmp (ep->key, key))
			return ep->value;

	return NULL;
}

void SetKeyValue (int iEntity, char *key, char *value)
{
	epair_t	*ep;
	
	for (ep=map.rgEntities[iEntity].epairs ; ep ; ep=ep->next)
		if (!strcmp (ep->key, key) )
		{
			FreeMem(ep->value, OTHER, strlen(ep->value) + 1);
			ep->value = copystring(value);
			return;
		}
	ep = (epair_t *)AllocMem(OTHER, sizeof(epair_t));
	ep->next = map.rgEntities[iEntity].epairs;
	map.rgEntities[iEntity].epairs = ep;
	ep->key = copystring(key);
	ep->value = copystring(value);
}


void GetVectorForKey(int iEntity, char *szKey, vec3_t vec)
{
	char	*k;
	double	v1, v2, v3;

	k = ValueForKey (iEntity, szKey);
	v1 = v2 = v3 = 0;
	// scanf into doubles, then assign, so it is vec_t size independent
	sscanf (k, "%lf %lf %lf", &v1, &v2, &v3);
	vec[0] = v1;
	vec[1] = v2;
	vec[2] = v3;
}


void WriteEntitiesToString (void)
{
	char *pCur;
	epair_t	*ep;
	char szLine[129];
	int	iEntity;
	int cLen;
	
	map.cTotal[BSPENT] = 0;

	for (iEntity=0; iEntity<map.cEntities; iEntity++)
	{
		ep = map.rgEntities[iEntity].epairs;

		// ent got removed
		if (!ep)
		{
			map.rgEntities[iEntity].cEntdata = 0;
			map.rgEntities[iEntity].pEntdata = NULL;
			continue;
		}
		
		cLen = 0;
		while (ep)
		{
			int i = strlen(ep->key) + strlen(ep->value) + 6;
			if (i <= 128)
				cLen += i;
			else
				cLen += 128;
			ep = ep->next;
		}

		// Add 4 for {\n and }\n
		cLen += 4;

		map.rgEntities[iEntity].cEntdata = cLen;
		map.cTotal[BSPENT] += cLen;
		map.rgEntities[iEntity].pEntdata = pCur = (char *)AllocMem(BSPENT, cLen);
		*pCur = 0;

		strcat(pCur, "{\n");
		pCur += 2;

		for (ep = map.rgEntities[iEntity].epairs; ep; ep=ep->next)
		{
			// Limit on Quake's strings of 128 bytes
			sprintf(szLine, "\"%.*s\" \"%.*s\"\n", MAX_KEY, ep->key, 122-strlen(ep->key), ep->value);
			strcat(pCur, szLine);
			pCur += strlen(szLine);
		}

		// No terminating null on this string
		pCur[0] = '}';
		pCur[1] = '\n';
	}
}

